'use strict';

/* --------------------------------------------------------------
 menu_trigger.js 2016-04-22
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Menu Trigger Controller
 *
 * This controller will handle the main menu trigger. Provide the "data-menu_trigger-menu-selector" attribute
 * that must select the main menu. It also works with the "user_configuration_service" so the user ID is required.
 * Provide it with the "data-menu_trigger-customer-id" attribute.
 *
 * There are three states for the main menu: "collapse", "expand" and "expand-all".
 */
gx.controllers.module('menu_trigger', ['user_configuration_service'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Main Menu Selector
     *
     * @type {jQuery}
     */
    var $menu = $(data.menuSelector);

    /**
     * Menu Items List Selector
     *
     * @type {jQuery}
     */
    var $list = $menu.find('nav > ul');

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Set the menu state.
     *
     * This function will update the UI and save the state in the users_configuration db table.
     *
     * @param {String} state Accepts the "collapse", "expand" and "expandAll".
     * @param {Boolean} save Optional (false), whether to save the change with the user configuration service.
     */
    function _setMenuState(state) {
        var save = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

        var stateClass = '';

        switch (state) {
            case 'collapse':
            case 'expand':
                stateClass = state;
                break;

            case 'expandAll':
                stateClass = 'expand-all';
                break;

            default:
                stateClass = 'expand';
        }

        var $radio = $this.find('input:radio#menu-' + stateClass);

        // Set the class to the <ul> element of the main menu.
        $list.attr('class', stateClass);

        // Make sure the correct radio is selected.
        $radio.prop('checked', true);

        // Display the next-state icons.
        $radio.prev('label').hide();
        if ($radio.next('label').length > 0) {
            $radio.next('label').show();
        } else {
            $this.find('label:first').show();
        }

        // Save the configuration setting.
        if (save) {
            jse.libs.user_configuration_service.set({
                data: {
                    userId: data.customerId,
                    configurationKey: 'menuVisibility',
                    configurationValue: state
                },
                onSuccess: function onSuccess(response) {
                    // Trigger a window resize in order to update the position of other UI elements.
                    $(window).trigger('resize');
                }
            });
        }
    }

    /**
     * Handles the radio buttons change.
     *
     * This is triggered by the click on the menu trigger button.
     */
    function _onInputRadioChange() {
        _setMenuState($(this).val(), true);
    }

    // ------------------------------------------------------------------------
    // INITIALIZE
    // ------------------------------------------------------------------------

    module.init = function (done) {
        _setMenuState(data.menuVisibility);

        $this.on('change', 'input:radio', _onInputRadioChange);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
